/*
 * Copyright (C) 2007 The Android Open Source Project
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package com.android.ide.common.resources;

import com.android.ide.common.rendering.api.ResourceValue;
import com.android.ide.common.resources.configuration.Configurable;
import com.android.ide.common.resources.configuration.FolderConfiguration;
import com.android.io.IAbstractFile;
import com.android.resources.ResourceType;

import java.util.Collection;

/**
 * Represents a Resource file (a file under $Project/res/)
 */
public abstract class ResourceFile implements Configurable {

    private final IAbstractFile mFile;
    private final ResourceFolder mFolder;

    protected ResourceFile(IAbstractFile file, ResourceFolder folder) {
        mFile = file;
        mFolder = folder;
    }

    protected abstract void load(ScanningContext context);
    protected abstract void update(ScanningContext context);
    protected abstract void dispose(ScanningContext context);

    @Override
    public FolderConfiguration getConfiguration() {
        return mFolder.getConfiguration();
    }

    /**
     * Returns the IFile associated with the ResourceFile.
     */
    public final IAbstractFile getFile() {
        return mFile;
    }

    /**
     * Returns the parent folder as a {@link ResourceFolder}.
     */
    public final ResourceFolder getFolder() {
        return mFolder;
    }

    public final ResourceRepository getRepository() {
        return mFolder.getRepository();
    }

    /**
     * Returns whether the resource is a framework resource.
     */
    public final boolean isFramework() {
        return mFolder.getRepository().isFrameworkRepository();
    }

    /**
     * Returns the list of {@link ResourceType} generated by the file. This is never null.
     */
    public abstract Collection<ResourceType> getResourceTypes();

    /**
     * Returns whether the file generated a resource of a specific type.
     * @param type The {@link ResourceType}
     */
    public abstract boolean hasResources(ResourceType type);

    /**
     * Returns the value of a resource generated by this file by {@link ResourceType} and name.
     * <p/>If no resource match, <code>null</code> is returned.
     * @param type the type of the resource.
     * @param name the name of the resource.
     */
    public abstract ResourceValue getValue(ResourceType type, String name);

    @Override
    public String toString() {
        return mFile.toString();
    }
}

